/*
 * Protocol independent interface for GM
 */
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <pthread.h>
#include <ctype.h>
#include <poll.h>

#include "gm.h"

/* counters exported in 2.x.24 */
#if GM_API_VERSION >= 0x20118 || \
	(GM_API_VERSION < 0x20100 && GM_API_VERSION >= 0x20018)
#include "gm_counters.h"
#endif

#include "libfma.h"
#include "libmyri.h"

/* Before gm_get_nic_info() exists, we need to REALLY cheat to get
 * the serial number, etc.
 */
#define _HAVE_GM_GET_INFO 0

#if !_HAVE_GM_GET_INFO
#include "gm_get_info_hack.h"
#endif

/* Node types that may be on the fabric */
enum gm_node_type
{
  GM_NODE_TYPE_GM = 0,          /* GM node */
  GM_NODE_TYPE_XM = 1,          /* XM node */
  GM_NODE_TYPE_MX = 2,          /* MX node */
  GM_NODE_TYPE_UNKNOWN = 98,    /* unknown node type */
  GM_NODE_TYPE_PURGED = 99      /* used to track "purged" nodes */
};


#define MYRI_GM_EVENT_TIMEOUT 1000	/* one second */
#define MYRI_GM_MAX_NICS 8
#define MYRI_GM_TX_BUF_LEN 2048		/* larger than RX to allow for route */
#define MYRI_GM_RX_BUF_LEN 1024

struct myri_gm_raw_send {
  int port;
  void *route;
  int route_len;
  void *tx_buf;
  int tx_len;
  void *context;

  void *raw_tx_buf;
  struct myri_gm *mip;
  int free_route;		/* if true, free route buffer when complete */

  struct myri_gm_raw_send *next;
  struct myri_gm_raw_send *prev;
};

/*
 * Used to link raw receive buffers together
 */
struct fm_raw_tx_buf {
  struct fm_raw_tx_buf *next;
};

struct myri_gm {
  int rd_fd;
  int wr_fd;

  struct gm_port *raw_port;
  unsigned int num_ports;

  pthread_mutex_t event_queue_lock;
  struct myri_event event_queue;
  struct myri_event null_event;		/* used for returning NO_EVENT */

  /* send_queue_lock guards both send_queue and raw_tx_buf_list */
  pthread_mutex_t send_queue_lock;
  struct myri_gm_raw_send send_queue;

  /* all raw registered tx buffers linked together */
  struct fm_raw_tx_buf *raw_tx_buf_list;

  pthread_t raw_thread;

  volatile int done;			/* set to tell waiter thread to exit */

  int sends_queued;
  int sends_pending;
  int sends_sent;
};

/* global vairables for the mi interface */
static struct myri_gm Myri[MYRI_GM_MAX_NICS];
static int GM_initted;

/*
 * Secret GM prototypes
 */
struct gm_mapper_state {
  gm_u8_t mapper_mac[6];
  gm_u8_t pad_after_mac[2];
  gm_u32_t map_version;
  gm_u32_t num_hosts;
  gm_u32_t network_configured;    /* Boolean */
  gm_u32_t routes_valid; /* Boolean */
  gm_u32_t level; /* mapper level */
  gm_u32_t flags; /* mapper flags */
};
gm_status_t _gm_mapper_open(struct gm_port **p, int nic_id, int api);
void _gm_provide_raw_receive_buffer(struct gm_port *p, void *buf);
void _gm_raw_send_with_callback(struct gm_port *p, void *buf, int total_len,
    int route_len, gm_send_completion_callback_t callback, void *context);
gm_status_t _gm_clear_all_routes(struct gm_port *p);
gm_status_t _gm_set_route(struct gm_port *p, unsigned char *remote_mac,
                             int route_len, unsigned char *route);

#if GM_API_VERSION > 0x200
gm_status_t _gm_set_mapper_state(struct gm_port *port,
     struct gm_mapper_state *mapper_state);
void _gm_set_host_name(struct gm_port *p, char *hostname);
#endif

#if GM_API_VERSION >= 0x20100
gm_status_t _gm_get_num_packet_interfaces(struct gm_port *p,
                                          unsigned int *result);
void _gm_raw_send_ex(struct gm_port *p, void *buf, int total_len, int route_len,
    gm_send_completion_callback_t callback, void *context, int port);
gm_status_t _gm_set_route_ex(struct gm_port *p, unsigned char *remote_mac,
                             int route_len, unsigned char *route,
			     int port, int route_num);
gm_status_t _gm_set_mapper_state_ex(struct gm_port *p,
     struct gm_mapper_state *mapper_state, unsigned int port);
#endif

#if defined(GM_API_VERSION_2_0_12)
gm_status_t _gm_set_node_type(struct gm_port *port,
    const gm_u8_t unique_id[6], gm_u32_t node_type);
#endif


/*
 * local prototypes
 */
static void myri_gm_enqueue_event(struct myri_gm *mip,
                                 struct myri_event *mep);
static void *myri_gm_raw_thread(void *arg);
static void myri_gm_release_raw_tx_buf(struct myri_gm *mip, void *p);
static void myri_gm_raw_send_callback(struct gm_port *port, void *v,
				     gm_status_t rc);
static void myri_gm_do_raw_send(struct myri_gm_raw_send *sp);
static int myri_open_sockets(struct myri_gm *mip);

/*
 * Instantiate the myri_ interface for GM
 */
int
myri_open(
  int nic_id)
{
  struct myri_gm *mip;
  gm_status_t gmrc;
  int rc;
  void *p;
  int ntok;
  int i;

  /* call gm_init() once */
  if (GM_initted == 0) {
    gmrc = gm_init();
    if (gmrc != GM_SUCCESS) {
       gm_perror ("gm_init()", gmrc);
       exit (1);
    }
    GM_initted = 1;
  }

  /* validate the NIC ID */
  if (nic_id >= MYRI_GM_MAX_NICS) {
    fprintf(stderr, "Bad nic_id: %d\n", nic_id);
    errno = EINVAL;
    return -1;
  }

  /* get pointer to interface info */
  mip = &Myri[nic_id];

  /* open the raw GM interface */
  gmrc = _gm_mapper_open(&mip->raw_port, nic_id, GM_API_VERSION);
  if (gmrc != GM_SUCCESS) {
    if (gmrc == GM_BUSY) {
      errno = EBUSY;
    } else if (gmrc == GM_PERMISSION_DENIED) {
      errno = EPERM;
    } else {
      errno = ENODEV;
    }
    return -1;
  }

  /* fill in information about this Myrinet interface */
  mip->rd_fd = -1;

  /* empty the event queue */
  mip->event_queue.next = &mip->event_queue;
  mip->event_queue.prev = &mip->event_queue;

  /* empty the send queue */
  mip->send_queue.next = &mip->send_queue;
  mip->send_queue.prev = &mip->send_queue;

  /* initialize null event */
  mip->null_event.type = MYRI_EVENT_NO_EVENT;

  /* set up the pipe we will be using for notifying of events */
  rc = myri_open_sockets(mip);
  if (rc == -1) {
    perror("opening pipe");
    return -1;
  }

  /* allocate locks now - gm_send_completion_callback_t needs them */
  pthread_mutex_init(&mip->event_queue_lock, NULL);
  pthread_mutex_init(&mip->send_queue_lock, NULL);

  /* Pre-allocate all the send buffers we'll actually use */
  ntok = gm_num_send_tokens(mip->raw_port);
  for (i=0; i<ntok; ++i) {
    p = gm_dma_malloc(mip->raw_port, MYRI_GM_TX_BUF_LEN);
    if (p == NULL) {
      fprintf(stderr, "gm_dma_malloc failed");
      exit(1);
    }
    myri_gm_release_raw_tx_buf(mip, p);
  }

  /* pre-allocate and provide raw receive buffers */
  ntok = gm_num_receive_tokens(mip->raw_port);
  for (i=0; i<ntok; ++i) {
    p = gm_dma_malloc(mip->raw_port, MYRI_GM_TX_BUF_LEN);
    if (p == NULL) {
      fprintf(stderr, "gm_dma_malloc failed");
      exit(1);
    }
    _gm_provide_raw_receive_buffer(mip->raw_port, p);
  }

  /* start the raw thread */
  rc = pthread_create(&mip->raw_thread, NULL, myri_gm_raw_thread, mip);
  if (rc != 0) {
    perror("Spawning raw thread");
    return -1;
  }

  /* fill in number of ports */
#if GM_API_VERSION < 0x20100
  mip->num_ports = 1;
#else
  (void) _gm_get_num_packet_interfaces(mip->raw_port, &mip->num_ports);
#endif

  /* use the nic_id as a handle for future calls */
  return nic_id;
}

static int
myri_open_sockets(
  struct myri_gm *mip)
{
  int sock;
  struct sockaddr_in addr;
  socklen_t len;
  int rc;

  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock == -1) {
    perror("opening socket");
    goto abort_with_nothing;
  }
  addr.sin_family = AF_INET;
  addr.sin_port = 0;
  addr.sin_addr.s_addr = INADDR_ANY;
  rc = bind(sock, (struct sockaddr*)&addr, sizeof(addr));
  if (rc == -1) {
    perror("binding socket");
    goto abort_with_sock;
  }
  rc = listen(sock, 1);
  if (rc == -1) {
    perror("listening");
    goto abort_with_sock;
  }
  mip->wr_fd = socket(PF_INET, SOCK_STREAM, 0);
  if (mip->wr_fd == -1) {
    perror("opening writer");
    goto abort_with_sock;
  }
  len = sizeof(addr);
  rc = getsockname(sock, (struct sockaddr*)&addr, &len);
  if (rc == -1) {
    perror("getting sock name");
    goto abort_with_writer;
  }
  rc = connect(mip->wr_fd, (struct sockaddr*)&addr, sizeof(addr));
  if (rc == -1) {
    perror("connecting writer");
    goto abort_with_writer;
  }
  mip->rd_fd = accept(sock, NULL, NULL);
  if (mip->rd_fd == -1) {
    perror("connecting reader");
    goto abort_with_writer;
  }
  close(sock);
  return 0;

 abort_with_writer:
  close(mip->wr_fd);
 abort_with_sock:
  close(sock);
 abort_with_nothing:
  return -1;
}

/*
 * Put a raw receive buffer on the list of available buffers
 * If there is a send waiting, send it now.
 * When in multi-threaded mode, this is called with send_queue_lock held.
 */
static void
myri_gm_release_raw_tx_buf(
  struct myri_gm *mip,
  void *p)
{
  struct fm_raw_tx_buf *tp;
  struct myri_gm_raw_send *sp;

  /* if send queue not empty, grab the oldest */
  sp = mip->send_queue.prev;
  if (sp != &mip->send_queue) {

    /* found a send to do, unlink it */
    sp->prev->next = sp->next;
    sp->next->prev = sp->prev;

    --sp->mip->sends_queued;

  /* no pending sends, put the tx_bug on the free list */
  } else {
    sp = NULL;

    tp = p;
    tp->next = mip->raw_tx_buf_list;
    mip->raw_tx_buf_list = tp;
  }

  /* if sp non-NULL, we have a send to perform */
  if (sp != NULL) {
    sp->raw_tx_buf = p;
    myri_gm_do_raw_send(sp);
  }
}

/*
 * Perform the guts of a raw send
 */
static void
myri_gm_do_raw_send(
  struct myri_gm_raw_send *sp)
{

  ++sp->mip->sends_pending;
  ++sp->mip->sends_sent;

  memcpy(sp->raw_tx_buf, sp->route, sp->route_len);
  memcpy(sp->raw_tx_buf+sp->route_len, sp->tx_buf, sp->tx_len);
#if GM_API_VERSION < 0x20100
  _gm_raw_send_with_callback(sp->mip->raw_port, 
                      sp->raw_tx_buf, sp->tx_len+sp->route_len, sp->route_len,
		      myri_gm_raw_send_callback, sp);
#else
  _gm_raw_send_ex(sp->mip->raw_port, 
                      sp->raw_tx_buf, sp->tx_len+sp->route_len, sp->route_len,
		      myri_gm_raw_send_callback, sp, sp->port);
#endif

  /* if message was copied, free it */
  if (sp->free_route) free(sp->route);
}

/*
 * Close a myri instance
 */
void
myri_close(
  int nic_id)
{
  struct myri_gm *mip;
  gm_alarm_t alarm;

  if (nic_id >= MYRI_GM_MAX_NICS) return;

  mip = &Myri[nic_id];
  if (mip->rd_fd != -1) {
    close(mip->rd_fd);
    close(mip->wr_fd);
  }
  
  mip->done = 1;

  /* this causes the receiving thread to wake up and notice
     that it is done */
  gm_initialize_alarm(&alarm);
  gm_set_alarm(mip->raw_port, &alarm, 1, NULL, NULL);

  /* wait for the raw thread to notice and exit */
  pthread_join(mip->raw_thread, NULL);

  if (mip->raw_port != NULL) gm_close(mip->raw_port);
}

/*
 * Return the file descriptor associated with a myri handle
 */
int
myri_fd(
  int nic_id)
{
  return Myri[nic_id].rd_fd;
}

/*
 * Get the next event from the GM thread
 */
int
myri_next_event(
  int nic_id,
  struct myri_event **rep,
  int timeout)
{
  struct myri_event *mep;
  struct myri_gm *mip;
  char junk;
  int rc;

  mip = &Myri[nic_id];

  /*
   * If a non-negative timeout is specified, poll on the pipe for
   * that long and return NO_EVENT if timeout occurs
   */
  if (timeout >= 0) {
    struct pollfd pf;

    /* execute a poll() with appropriate timeout */
    pf.fd = mip->rd_fd;
    pf.events = POLLIN;
    rc = poll(&pf, 1, timeout);

    /* bad if error and not EINTR */
    if (rc == -1 && errno != EINTR
#if HAVE_DECL_ERESTART
	&& errno != ERESTART
#endif
	&& errno != EAGAIN) {
      return -1;
    }

    /* return of 0 from poll means nothing is ready */
    if (rc == 0 || (pf.revents & POLLIN) == 0) {
      *rep = &mip->null_event;
      return 0;
    }
  }

  /* read one byte from the pipe to ensure an event is ready */
  rc = recv(mip->rd_fd, &junk, 1, 0);
  if (rc != 1) return -1;

  /* lock while we manipulate the queue */
  pthread_mutex_lock(&mip->event_queue_lock);

  mep = mip->event_queue.prev;
  if (mep == &mip->event_queue) {
    fprintf(stderr, "event queue unexpectedly empty!\n");
    return -1;
  }
  
  /* unlink this element from the list*/
  mep->next->prev = mep->prev;
  mep->prev->next = mep->next;

  /* all done with the queue */
  pthread_mutex_unlock(&mip->event_queue_lock);
  
  *rep = mep;
  return 0;
}

/*
 * Enqueue an event and let the main thread know about it
 */
static void
myri_gm_enqueue_event(
  struct myri_gm *mip,
  struct myri_event *mep)
{
  char junk;
  int rc;

  /* lock while we manipulate the queue */
  pthread_mutex_lock(&mip->event_queue_lock);

  /* link this element into the head of the list */
  mep->next = mip->event_queue.next;
  mep->prev = &mip->event_queue;

  mep->next->prev = mep;
  mep->prev->next = mep;

  /* all done with the queue */
  pthread_mutex_unlock(&mip->event_queue_lock);

  /* write a byte to the pipe to wake up main thread */
  rc = send(mip->wr_fd, &junk, 1, 0);
  if (rc != 1) {
    perror("Writing to pipe");
    exit(1);
  }
}

/*
 * raw communication thread
 */
static void *
myri_gm_raw_thread(
  void *arg)
{
  struct myri_gm *mip;
  struct myri_event *mep;
  union gm_recv_event *event;
  void *rx_buf;
  void *p;

  mip = arg;
  rx_buf = NULL;
  mep = NULL;

  /*
   * Loop waiting for events.  Everytime we get an event, we queue
   * a corresponding event for the main thread to read and write into
   * our end of the pipe to inidicate it's time to look for an event.
   */
  while (mip->done == 0) {

    /* get a struct to hold our next event */
    if (mep == NULL) {
      mep = (struct myri_event *) malloc(sizeof(*mep));
      if (mep == NULL) {
	perror("allocating raw event struct");
	exit(1);
      }
    }

    /* get a receive buffer now if we need one */
    if (rx_buf == NULL) {
      rx_buf = malloc(MYRI_GM_RX_BUF_LEN);
      if (rx_buf == NULL) {
	perror("allocating raw receive buffer");
	exit(1);
      }
    }

    /*
     * get the next event, holding send_queue lock to serialize with 
     * anyone trying to do a new raw_send.  The only reason this does not
     * livelock is because gm_blocking_receive does not *actually* block -
     * gm_unknown() is where that is done.
     */
    pthread_mutex_lock(&mip->send_queue_lock);
    event = gm_blocking_receive(mip->raw_port);
    pthread_mutex_unlock(&mip->send_queue_lock);

    switch (GM_RECV_EVENT_TYPE (event)) {

      /* incoming receive */
      case GM_RAW_RECV_EVENT:
	p = gm_ntohp (event->recv.buffer);

	mep->type = MYRI_EVENT_RAW_RECV_COMPLETE;
	mep->d.raw_recv.rxbuf = rx_buf;
	mep->d.raw_recv.rx_len = gm_ntoh_u32(event->recv.length);
#if GM_API_VERSION < 0x20100
	mep->d.raw_recv.port = 0;
#else
	mep->d.raw_recv.port = gm_ntoh_u8(event->recv.raw_packet_interface);
#endif

	/* copy over the data */
	memcpy(rx_buf, p, mep->d.raw_recv.rx_len);

	/* re-arm the raw receive */
	_gm_provide_raw_receive_buffer(mip->raw_port, p);

	/* pass along the event */
	myri_gm_enqueue_event(mip, mep);

	/* both these need re-allocation */
	mep = NULL;
	rx_buf = NULL;
	break;

#if GM_API_VERSION >= 0x2000b
      /* handle this - report it back to caller */
      case GM_FATAL_FIRMWARE_ERROR_EVENT:
	fprintf(stderr, "NIC Firmware error\n");
	exit(1);
#endif

      default:
	gm_unknown(mip->raw_port, event);
	break;
    }
  }

  return NULL;
}

/*
 * Callback for raw send completion
 */
static void
myri_gm_raw_send_callback(
  struct gm_port *port,
  void *v,
  gm_status_t rc)
{
  struct myri_gm *mip;
  struct myri_event *mep;
  struct myri_gm_raw_send *sp;

  sp = v;
  mip = sp->mip;

  /* allocate an event struct */
  mep = (struct myri_event *) malloc(sizeof(*mep));
  if (mep == NULL) {
    perror("allocating event struct");
    exit(1);
  }

  /* fill in and enqueue the event */
  mep->type = MYRI_EVENT_RAW_SEND_COMPLETE;
  mep->d.raw_send.context = sp->context;
  myri_gm_enqueue_event(mip, mep);

  /* Now, release this tx buffer back to the pool (which may trigger a new
   * send)
   */
  pthread_mutex_lock(&mip->send_queue_lock);
  --mip->sends_pending;
  myri_gm_release_raw_tx_buf(mip, sp->raw_tx_buf);
  pthread_mutex_unlock(&mip->send_queue_lock);

  /* free the send queue struct */
  free(sp);
}

/*
 * Release an event struct returned by myri_next_event
 */
void
myri_release_event(
  struct myri_event *mep)
{
  /* NO_EVENT is passed via a static event buffer */
  if (mep->type == MYRI_EVENT_NO_EVENT) {
    return;
  }

  if (mep->type == MYRI_EVENT_RAW_RECV_COMPLETE) {
    free(mep->d.raw_recv.rxbuf);
  }
  free(mep);
}

/*
 * Perform a raw send
 */
int
myri_raw_send(
  int nic_id,
  int port,
  void *route,
  int route_len,
  void *txbuf,
  int txlen,
  void *context)
{
  struct myri_gm *mip;
  struct myri_gm_raw_send *sp;
  struct fm_raw_tx_buf *tp;
  int ret;

  ret = 0;			/* default is good return */
  mip = &Myri[nic_id];

  /* allocate a send queue entry */
  sp = (struct myri_gm_raw_send *) calloc(sizeof(*sp), 1);
  if (sp == NULL) {
    perror("Error allocating send queue entry");
    return -1;
  }

  /* fill it in */
  sp->mip = mip;
  sp->port = port;
  sp->route_len = route_len;
  sp->tx_len = txlen;
  sp->context = context;

  /*
   * Now we need to look at the raw_tx_buffer list.  If anything on it,
   * we can send now.  Otherwise, we need to queue this.
   */
  pthread_mutex_lock(&mip->send_queue_lock);

  tp = mip->raw_tx_buf_list;

  /* If we got a buffer, take it off the list */
  if (tp != NULL) {
    mip->raw_tx_buf_list = tp->next;

    /* fill in buffer pointers */
    sp->route = route;
    sp->tx_buf = txbuf;
    sp->raw_tx_buf = tp;
    sp->free_route = 0;

    myri_gm_do_raw_send(sp);

  /* no raw tx buffers available, queue this send */
  } else {
    sp->route = malloc(route_len + txlen);
    if (sp->route == NULL) {
      free(sp);
      perror("Allocating send buffer");
      ret = -1;
      goto leave;
    }
    sp->tx_buf = sp->route+route_len;
    memcpy(sp->route, route, route_len);
    memcpy(sp->tx_buf, txbuf, txlen);
    sp->free_route = 1;

    sp->next = mip->send_queue.next;
    sp->prev = &mip->send_queue;

    sp->next->prev = sp;
    sp->prev->next = sp;

    ++sp->mip->sends_queued;
  }
 leave:
  pthread_mutex_unlock(&mip->send_queue_lock);

  return ret;
}

/*
 * Get the hostname of a remote host.  If the hostname is returned as ending
 * in ":X" then trim the ":X" and use X and remote nic_id, else remote nic_id
 * is 0.
 */
int
myri_mac_to_hostname(
  int nic_id,
  lf_mac_addr_t mac_addr,
  lf_string_t hostname,
  int *his_nic_id)
{
  gm_status_t gmrc;
  unsigned int node_id;
  int hisid;
  int len;
#if GM_API_VERSION < 0x200
  char *gm_hostname;
#else
  char gm_hostname[GM_MAX_HOST_NAME_LEN+1];
#endif
  struct myri_gm *mip;

  mip = &Myri[nic_id];
  
  gmrc = gm_unique_id_to_node_id(mip->raw_port, (char *)mac_addr, &node_id);
  if (gmrc != GM_SUCCESS) {
    return -1;
  }

#if GM_API_VERSION < 0x200
  gm_hostname = gm_node_id_to_host_name(mip->raw_port, node_id);
  if (gm_hostname == NULL) {
    return -1;
  }
#else
  gmrc = gm_node_id_to_host_name_ex(mip->raw_port, 2000, node_id, &gm_hostname);
  if (gmrc != GM_SUCCESS) {
    return -1;
  }
#endif


  len = strlen(gm_hostname);

  /* Treat a NULL hostname as a failure */
  if (len == 0) {
    return -1;
  }

  /* Check for trailing ":X" */
  if (gm_hostname[len-2] == ':' && isdigit(gm_hostname[len-1])) {
    gm_hostname[len-2] = '\0';
    hisid = atoi(gm_hostname + len - 1);
  } else {
    hisid = 0;
  }
  if (his_nic_id != NULL) {
    *his_nic_id = hisid;
  }
  
  strncpy(hostname, gm_hostname, LF_STRING_LEN);
  return 0;
}

/*
 * Return information about this NIC
 */
int
myri_get_nic_info(
  int nic_id,
  struct myri_nic_info *nip)
{
  struct myri_nic_info info;
  gm_status_t gmrc;
  struct myri_gm *mip;
  struct gm_nic_info nic_info;

  mip = &Myri[nic_id];

  /* get MAC address */
  gmrc = gm_get_unique_board_id(mip->raw_port, (char *)&info.mac_addr);
  if (gmrc != GM_SUCCESS) {
    return -1;
  }

#if GM_API_VERSION < 0x20100
  info.num_ports = 1;
#else
  gmrc = _gm_get_num_packet_interfaces(mip->raw_port, &info.num_ports);
  if (gmrc != GM_SUCCESS) {
    return -1;
  }
#endif

  /* speed is always 2 on GM */
  info.speed = MYRI_SPEED_2G;

#if GM_API_VERSION < 0x20100
  info.num_routes = 1;
#else
  info.num_routes = 8;
#endif
  info.num_routes_is_per_port = FALSE;

  /* get info about this NIC */
  gmrc = gm_get_nic_info(mip->raw_port, &nic_info);
  if (gmrc != GM_SUCCESS) {
    return -1;
  }

  strcpy(info.serial_no, nic_info.serial_no);
  strcpy(info.product_id, nic_info.product_id);

  /* copy out the info we gathered */
  *nip = info;
  return 0;
}

/*
 * No prep needed
 */
int myri_set_route_start(
  int nic_id)
{
  return 0;
}

/*
 * Set the mapper state at this time.
 */
int myri_set_route_end(
  int nic_id,
  lf_mac_addr_t mapper_mac,
  int map_version,
  int num_hosts)
{
  struct gm_mapper_state ms;
  struct myri_gm *mip;
  gm_status_t gmrc;

  mip = &Myri[nic_id];

  /* fill in map status */
  memset(&ms, 0, sizeof(ms));
  LF_MAC_COPY(ms.mapper_mac, mapper_mac);
  ms.num_hosts = num_hosts;
  ms.network_configured = 1;
  ms.map_version = map_version;

  /* set the mapper status in GM */

  /* for old GM, no port arg */
#if GM_API_VERSION < 0x20100
  gmrc = _gm_set_mapper_state (mip->raw_port, &ms);
  if (gmrc != GM_SUCCESS) {
    return -1;
  }
#else
  /* newer GM, set mapper state for each port */
  {
    int port;
    for (port = 0; port < mip->num_ports; ++port) {
      gmrc = _gm_set_mapper_state_ex (mip->raw_port, &ms, port);
      if (gmrc != GM_SUCCESS) {
	return -1;
      }
    }
  }
#endif

  return 0;
}

/*
 * Set a route
 */
int
myri_set_route(
  int nic_id,
  lf_mac_addr_t remote_mac,
  enum lf_firmware_type node_type,
  int remote_port,
  int route_num,
  int local_port,
  unsigned char *route,
  int route_len)
{
  gm_status_t gmrc;
  struct myri_gm *mip;

  mip = &Myri[nic_id];

#if GM_API_VERSION < 0x20100
  gmrc = _gm_set_route(mip->raw_port, remote_mac, route_len, route);
#else
  gmrc = _gm_set_route_ex(mip->raw_port, remote_mac, route_len, route,
                          local_port, route_num);
#endif

  /* propagate sucess/failure */
  if (gmrc != GM_SUCCESS) {
    errno = EINVAL;
    return -1;
  }

#if defined(GM_API_VERSION_2_0_12)
  {
    int gm_node_type;

    switch (node_type) {
      case MYRI_FW_GM_1:
      case MYRI_FW_GM_2:
	gm_node_type = GM_NODE_TYPE_GM;
	break;
      case MYRI_FW_MX_1:
	gm_node_type = GM_NODE_TYPE_MX;
	break;
      case MYRI_FW_XM_1:
      case MYRI_FW_2Z_1:
	gm_node_type = GM_NODE_TYPE_XM;
	break;
      case MYRI_FW_UNKNOWN:
      default:
	gm_node_type = GM_NODE_TYPE_UNKNOWN;
	break;
    }
    gmrc = _gm_set_node_type(mip->raw_port, remote_mac, gm_node_type);
    if (gmrc != GM_SUCCESS) {
      return -1;
    }
  }
#endif

  return 0;
}

int
myri_get_hostname(
  int nic_id,
  char *hostname)
{
  gm_status_t gmrc;
  struct myri_gm *mip;
  char name[GM_MAX_HOST_NAME_LEN+1];

  mip = &Myri[nic_id];

  /* get the hostname for this NIC */
  gmrc = gm_get_host_name(mip->raw_port, name);
  if (gmrc != GM_SUCCESS) {
    return -1;
  }

  /* Make sure we will have enough room for the name */
  if (strlen(name) > LF_STRING_LEN-1) {
    errno = ENOMEM;
    return -1;
  }

  /* copy out the hostname and return */
  strcpy(hostname, name);
  return 0;
}

int
myri_set_hostname(
  int nic_id,
  char *hostname)
{
  struct myri_gm *mip;

  mip = &Myri[nic_id];

  _gm_set_host_name(mip->raw_port, hostname);
  return 0;
}

/*
 * Set the default hostname for a NIC
 */
int
myri_set_dflt_hostname(
  int nic_id,
  char *hostname)
{
  lf_string_t name;
  int rc;

  if (nic_id == 0) {
    rc = myri_set_hostname(nic_id, hostname);
  } else {
    snprintf(name, sizeof(name), "%s:%d", hostname, nic_id);
    rc = myri_set_hostname(nic_id, name);
  }

  return rc;
}

/*
 * Return counters for a NIC port
 */
int
myri_get_nic_counters(
  int nic_id,
  int port,
  struct myri_nic_counters *counters)
{

#if GM_API_VERSION < 0x20018 || \
	(GM_API_VERSION >= 0x20100 && GM_API_VERSION < 0x20118)
  memset(counters, 0, sizeof(*counters));

#else
  struct gm_counters gm_counters;
  struct myri_gm *mip;

  mip = &Myri[nic_id];

  gm_get_counters(mip->raw_port, &gm_counters, port);

  counters->badcrcs = gm_counters.badcrc__invalid_crc8_cnt
		      + gm_counters.badcrc__misaligned_crc32_cnt
		      + gm_counters.badcrc__invalid_crc32_cnt;
  counters->tx_packets = gm_counters.netsend_cnt;
  counters->rx_packets = gm_counters.netrecv_cnt;
#endif
  
  return 0;
}

/*
 * Return firmware code
 */
enum lf_firmware_type
myri_firmware_type()
{
#if GM_API_VERSION < 0x200
  return MYRI_FW_GM_1;
#else
  return MYRI_FW_GM_2;
#endif
}

/*
 * set info for NIC auto-reply to scouts if supported
 */
int
myri_set_nic_reply_info(
  int nic_id,
  void *blob,
  int size)
{
  return 0;
}
